/* Compal Laptop Control
 * Author: Krzemin <pkrzemin@o2.pl>
 * WWW: http://krzemin.iglu.cz
 * This software is distributed under the GNU General Public License.
 */

#include "frmcompal.h"

frmCompal::frmCompal(QWidget * parent, Qt::WFlags f) : QWidget(parent, f)
{
	ui.setupUi(this);

	if(QSystemTrayIcon::isSystemTrayAvailable())
	{
		QPixmap pixmap(":/mask/mask.png");
		setMask(pixmap.mask());
	}

	ui.cbBluetooth->setDisabled(!cBtAvailable());
	ui.cbWifi->setDisabled(!cWifiAvailable());

	connect (ui.cbBluetooth, SIGNAL(toggled(bool)), this, SLOT(cbBluetoothToggled(bool)));
	connect (ui.cbWifi, SIGNAL(toggled(bool)), this, SLOT(cbWifiToggled(bool)));
	connect (ui.hsBrightness, SIGNAL(valueChanged(int)), this, SLOT(hsBrightnessChanged(int)));

	setMouseTracking(true);

	if(GlobalConfig().restoreSettings())
	{
		ui.cbBluetooth->setChecked(GlobalConfig().bluetoothOn());
		cbBluetoothToggled(ui.cbBluetooth->isChecked());
		ui.cbWifi->setChecked(GlobalConfig().wlanOn());
		cbWifiToggled(ui.cbWifi->isChecked());
		ui.hsBrightness->setValue(GlobalConfig().brightness());
		hsBrightnessChanged(ui.hsBrightness->value());
	}

	setWindowOpacity(GlobalConfig().opacity() ? 0.85 : 1);

	updateTimer = new QTimer(this);
	connect(updateTimer, SIGNAL(timeout()), this, SLOT(updateUiStatus()));
	updateTimer->start(1000);
}

void frmCompal::createTrayIcon()
{
	showAction = new QAction(tr("Show"), this);
	connect(showAction, SIGNAL(triggered()), this, SLOT(showHideWindow()));

	restoreAction = new QAction(tr("Save and restore settings"), this);
	restoreAction->setCheckable(true);
	restoreAction->setChecked(GlobalConfig().restoreSettings());
	connect(restoreAction, SIGNAL(triggered()), this, SLOT(restoreClicked()));

	opacityAction = new QAction(tr("Opacity window"), this);
	opacityAction->setCheckable(true);
	opacityAction->setChecked(GlobalConfig().opacity());
	connect(opacityAction, SIGNAL(triggered()), this, SLOT(opacityClicked()));

	aboutAction = new QAction(tr("About..."), this);
	connect(aboutAction, SIGNAL(triggered()), this, SLOT(showAbout()));

	quitAction = new QAction(tr("Quit"), this);
	connect(quitAction, SIGNAL(triggered()), qApp, SLOT(quit()));

	trayIconMenu = new QMenu(this);
	trayIconMenu->addAction(showAction);
	trayIconMenu->addAction(restoreAction);
	trayIconMenu->addAction(opacityAction);
	trayIconMenu->addAction(aboutAction);
	trayIconMenu->addSeparator();
	trayIconMenu->addAction(quitAction);

	trayIcon = new QSystemTrayIcon(this);
	trayIcon->setContextMenu(trayIconMenu);
	trayIcon->setIcon(QIcon(":/icon/computer-laptop.png"));

	connect(trayIcon, SIGNAL(activated(QSystemTrayIcon::ActivationReason)),
			this, SLOT(iconActivated(QSystemTrayIcon::ActivationReason)));

	connect(this, SIGNAL(toolTipChange(const QString &)), this, SLOT(setToolTip(const QString &)));

	trayIcon->show();
}

void frmCompal::setToolTip(const QString & tip)
{
	trayIcon->setToolTip(tip);
}

void frmCompal::updateUiStatus()
{
	bool bt = cBtStatus(), wifi = cWifiStatus();
	int lcd = cLCDStatus();

	ui.cbBluetooth->setChecked(bt);
	ui.cbWifi->setChecked(wifi);
	ui.hsBrightness->setSliderPosition(lcd);

	GlobalConfig().setBluetoothOn(bt);
	GlobalConfig().setWlanOn(wifi);
	GlobalConfig().setBrightness(lcd);
	GlobalConfig().save();

	QString toolTipStr = tr("Bluetooth") + QString(": ") + (bt ? tr("On") : tr("Off")) + "\n";
	toolTipStr += tr("Wireless LAN") + QString(": ") + (wifi ? tr("On") : tr("Off")) + "\n";
	toolTipStr += tr("LCD brightness:") + QString(" %1%").arg(lcd * 100 / 7);

	emit toolTipChange(toolTipStr);
}

void frmCompal::cbBluetoothToggled(bool checked)
{
	cSetBtStatus(checked);
}

void frmCompal::cbWifiToggled(bool checked)
{
	cSetWifiStatus(checked);
}

void frmCompal::hsBrightnessChanged(int value)
{
	cSetLCDStatus(value);
}

void frmCompal::iconActivated(QSystemTrayIcon::ActivationReason reason)
{
	if ( reason == QSystemTrayIcon::Trigger )
		showHideWindow();
}

void frmCompal::showHideWindow()
{
	int SW = QApplication::desktop()->width(),
		SH = QApplication::desktop()->height();

	move(QCursor::pos().x() - width() / 2, QCursor::pos().y() - height() - 10);

	int x = pos().x(),
		y = pos().y();

	const int SPACING = 30;

	if (x <= SPACING ) x = SPACING;
	if (y <= SPACING ) y = SPACING;
	if (x + width() > SW - SPACING ) x = SW - width() - SPACING;
	if (y + height() > SH - SPACING ) y = SH - height() - SPACING;

	move(x, y);

	isVisible() ? hide() : show();
}

void frmCompal::showAbout()
{
	about.show();
}

void frmCompal::restoreClicked()
{
	GlobalConfig().setRestoreSettings(restoreAction->isChecked());
	GlobalConfig().save();
}

void frmCompal::opacityClicked()
{
	setWindowOpacity(opacityAction->isChecked() ? 0.85 : 1);
	GlobalConfig().setOpacity(opacityAction->isChecked());
	GlobalConfig().save();
}

void frmCompal::wheelEvent(QWheelEvent *event)
{
	ui.hsBrightness->setValue(ui.hsBrightness->value() + (event->delta() > 0 ? 1 : -1));
}

void frmCompal::keyPressEvent(QKeyEvent *event)
{
	int change = 0;
	if( (event->key() == Qt::Key_Left) || (event->key() == Qt::Key_Down) )
		change = -1;
	else if( (event->key() == Qt::Key_Right) || (event->key() == Qt::Key_Up) )
		change = 1;

	ui.hsBrightness->setValue(ui.hsBrightness->value() + change);

	if(event->key() == Qt::Key_B)
	{
		ui.cbBluetooth->setChecked(!ui.cbBluetooth->isChecked());
		cbBluetoothToggled(ui.cbBluetooth->isChecked());
	}
	
	if(event->key() == Qt::Key_W)
	{
		ui.cbWifi->setChecked(!ui.cbWifi->isChecked());
		cbWifiToggled(ui.cbWifi->isChecked());
	}
}

void frmCompal::paintEvent(QPaintEvent *event)
{
	if(QSystemTrayIcon::isSystemTrayAvailable())
	{
		QPainter painter(this);
		painter.setPen(palette().color(QPalette::WindowText));
		painter.drawPolygon(borderPoints, 8);
		event->accept();
	}
}
